import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.io.*;
import java.net.*;
import java.util.*;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;

public class Client {

	public static void main(String args[]) {
		Gui gui = new Gui();
		Session session = new Session();

		// observers
		SessionObserver sobserver = new SessionObserver(session);
		GuiObserver gobserver = new GuiObserver(gui);

		gui.attach(sobserver);
		session.attach(gobserver);

		gui.draw();
	}
}

class Gui {
	SessionObserver sobserver;

	static int lineheight = 25;
	static int height = 300;
	static int width = 300;

	JFrame frame;
	JPanel guiPan, conPan, msgPan;
	JTextField host, port, message, name;
	JButton connect, submit;
	JTextArea transcript;
	JScrollPane tss;

	public void attach(SessionObserver sobserver) {
		this.sobserver = sobserver;
	}

	public void draw() {
		frame = new JFrame("TCP Chat Client");
		frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		frame.addWindowListener(new WindowAdapter() {
			public void windowClosing(java.awt.event.WindowEvent e) {
				try {
					sobserver.close();
				} catch (IOException e1) {
					System.err.println(e1.getMessage());
				} finally {
					e.getWindow().dispose();
				}
			}

		});

		guiPan = new JPanel();
		guiPan.setLayout(new BoxLayout(guiPan, BoxLayout.Y_AXIS));

		// connection
		conPan = new JPanel();
		conPan.setLayout(new BoxLayout(conPan, BoxLayout.X_AXIS));

		name = new JTextField("Nickname");
		name.setPreferredSize(new Dimension(width / 4, lineheight));
		conPan.add(name);

		host = new JTextField("localhost");
		host.setPreferredSize(new Dimension(width / 4, lineheight));
		conPan.add(host);

		port = new JTextField("12345");
		port.setPreferredSize(new Dimension(width / 6, lineheight));
		conPan.add(port);

		connect = new JButton("Connect");
		connect.setPreferredSize(new Dimension(width / 3, lineheight));
		connect.addActionListener(new Connect());
		conPan.add(connect);

		guiPan.add(conPan);

		// transcript
		transcript = new JTextArea();
		transcript.setEditable(false);
		transcript.setLineWrap(true);
		tss = new JScrollPane(transcript);
		tss.setPreferredSize(new Dimension(width, height));
		guiPan.add(tss);

		// messages
		msgPan = new JPanel();
		msgPan.setLayout(new BoxLayout(msgPan, BoxLayout.X_AXIS));

		message = new JTextField();
		message.setPreferredSize(new Dimension(width / 3 * 2, lineheight));
		message.addActionListener(new Submit());
		msgPan.add(message);

		submit = new JButton("Send");
		submit.setPreferredSize(new Dimension(width / 3, lineheight));
		submit.addActionListener(new Submit());
		msgPan.add(submit);

		// disable chat inputs
		message.setEnabled(false);
		submit.setEnabled(false);
		guiPan.add(msgPan);

		frame.add(guiPan);

		frame.pack();
		frame.setVisible(true);
	}

	public void update(String data) {
		transcript.append(data + "\n");
		JScrollBar scrollBar = tss.getVerticalScrollBar();
		scrollBar.setValue(scrollBar.getMaximum());
	}
	
	public void lock() {
		name.setEnabled(false);
		host.setEnabled(false);
		port.setEnabled(false);
		connect.setEnabled(false);
		message.setEnabled(false);
		submit.setEnabled(false);
	}

	class Connect implements ActionListener {
		public void actionPerformed(ActionEvent event) {
			String hostname = host.getText();
			String portnumber = port.getText();
			String nickname = name.getText();

			if (!hostname.equals("") && !portnumber.equals("")) {
				try {
					int portnr = Integer.parseInt(portnumber);
					transcript.append("Connecting to " + hostname + " on port " + portnr + "...\n");

					// start session
					sobserver.open(nickname, hostname, portnr);
					transcript.append("You're now connected to the server.\n");

					// disable connection inputs
					name.setEnabled(false);
					host.setEnabled(false);
					port.setEnabled(false);
					connect.setEnabled(false);

					// enable chat inputs
					message.setEnabled(true);
					submit.setEnabled(true);

				} catch (NumberFormatException e) {
					transcript.append("Portnumber must be numeric.\n");
				} catch (UnknownHostException e) {
					transcript.append("Sock: " + e.getMessage() + "\n");
				} catch (IOException e) {
					transcript.append("IO: " + e.getMessage() + "\n");
				}

			} else {
				transcript.append("Please provide hostname and portnumber.\n");
			}
		}
	}

	class Submit implements ActionListener {
		public void actionPerformed(ActionEvent event) {
			String msg = message.getText();
			if (!msg.equals("")) {
				try {
					sobserver.send(msg);
					message.setText(null);
				} catch (IOException e) {
					System.out.println(e.getMessage());
					try {
						sobserver.close();
					} catch (IOException e1) {
						System.out.println(e1.getMessage());
					}
				}
			}
		}
	}
}

class Session extends Thread {

	Socket socket;
	DataInputStream in;
	DataOutputStream out;
	GuiObserver gobserver;
	String nickname;

	public void run() {
		String data;
		try {
			while (true) {
				if (in.available() > 0) {
					data = in.readUTF();
					gobserver.update(data);
				}
			}
		} catch (IOException e) {
			System.out.println(e.getMessage());
		} finally {
			try {
				socket.close();
			} catch (IOException e) {
				System.out.println(e.getMessage());
			}
		}
		gobserver.update("The connection to the server is closed.\nRestart the application.");
		gobserver.lock();
	}

	public void open(String nickname, String host, int port) throws UnknownHostException, IOException {
		socket = new Socket(host, port);
		this.nickname = nickname;
		in = new DataInputStream(socket.getInputStream());
		out = new DataOutputStream(socket.getOutputStream());
		this.start();
	}

	public void attach(GuiObserver gobserver) {
		this.gobserver = gobserver;
	}

	public void send(String msg) throws IOException {
		out.writeUTF(nickname + ": " + msg);
	}

	public void close() throws IOException {
		if (socket != null)
			socket.close();
	}

}

class SessionObserver {

	private Session session;

	public SessionObserver(Session session) {
		this.session = session;
	}

	public void open(String nickname, String host, int port) throws UnknownHostException, IOException {
		session.open(nickname, host, port);
	}

	public void send(String msg) throws IOException {
		session.send(msg);
	}

	public void close() throws IOException {
		session.close();
	}

}

class GuiObserver {
	private Gui gui;

	public GuiObserver(Gui session) {
		this.gui = session;
	}

	public void update(String data) {
		gui.update(data);
	}
	public void lock() {
		gui.lock();
	}
}
